/* ecrypt-frogbit-prng.c */

#define TEST (0) /* setting this to non-zero creates a Berlekamp-Massey
                    validation of PRNGs -- requires source files from
                    another project*/

#include <string.h>

#if !TEST
#include "ecrypt-sync-ae.h"
#else
#include <stdio.h>
#include <limits.h>
#include "longpoly.h"
#include "lfsr.h"
#endif

#include "ecrypt-frogbit-prng.h"

/**************** Start of parameters that can be modified ****************/

#define FROGBIT_KEYSCHED_C1 (03333333)
#define FROGBIT_KEYSCHED_C2 (02626262)

/* PRNG design made of equally spaced selection of polynomials among the list */
static int prng_design[FROGBIT10][2]=
{{  9%MTGFSR_N13,  2%MTGFSR_N11}
,{  7%MTGFSR_N11, 26%MTGFSR_N13}
,{ 44%MTGFSR_N13, 12%MTGFSR_N11}
,{ 17%MTGFSR_N11, 62%MTGFSR_N13}
,{ 79%MTGFSR_N13, 22%MTGFSR_N11}
,{ 27%MTGFSR_N11, 97%MTGFSR_N13}
,{114%MTGFSR_N13, 32%MTGFSR_N11}
,{ 37%MTGFSR_N11,132%MTGFSR_N13}
,{150%MTGFSR_N13, 42%MTGFSR_N11}
,{ 47%MTGFSR_N11,167%MTGFSR_N13}
};

#if 3!=(((FROGBIT_KEYSCHED_C1>>20)\
        |(FROGBIT_KEYSCHED_C1>>18)\
        |(FROGBIT_KEYSCHED_C1>>16)\
        |(FROGBIT_KEYSCHED_C1>>14)\
        |(FROGBIT_KEYSCHED_C1>>12)\
        |(FROGBIT_KEYSCHED_C1>>10)\
        |(FROGBIT_KEYSCHED_C1>> 8)\
        |(FROGBIT_KEYSCHED_C1>> 6)\
        |(FROGBIT_KEYSCHED_C1>> 4)\
        |(FROGBIT_KEYSCHED_C1>> 2)\
        | FROGBIT_KEYSCHED_C1     \
        )&3)
#error "The modified key schedule must somehow preclude an all-zero state."
#endif

#if FROGBIT_KEYSCHED_C2&~((1<<(MTGFSR_T-1))-1)
#error "The key schedule must prevent a state with non-zero unused bits."
#endif

/**************** End of parameters that can be modified ****************/

/**************** Polynoms for TGFSR algorithm ****************/

/*
   The data tables below were generated by a separate program according to
   the TGFSR theory.

   Only 20 values are actually used from the tables, according to the indices
   found in the array prng_design (initialized above).
*/
struct lfsr_polynom_str {
   unsigned long Aj;
   #if TEST
   unsigned long Bj[2];
   #endif
};

#if TEST
#define DAT(wj,n,m,Aj,Bj_1,Bj_0,terms) {Aj,{Bj_0,Bj_1}},
#else
#define DAT(wj,n,m,Aj,Bj_1,Bj_0,terms) {Aj},
#endif

static const struct lfsr_polynom_str poly_table_11[MTGFSR_N11]={
#if (1==MTGFSR_M)
DAT(11,3,1,0x817,0x2,0xA802B85F,15)
DAT(11,3,1,0x82B,0x2,0xA8020A03,9)
DAT(11,3,1,0x863,0x2,0xA8064CEB,15)
DAT(11,3,1,0x89F,0x2,0xA8281077,13)
DAT(11,3,1,0x8E7,0x2,0xA82CE62F,17)
DAT(11,3,1,0x913,0x2,0xA902B91B,15)
DAT(11,3,1,0x95B,0x2,0xA906FFF3,21)
DAT(11,3,1,0x97F,0x2,0xA9065F17,17)
DAT(11,3,1,0x983,0x2,0xA928038B,13)
DAT(11,3,1,0x9C7,0x2,0xA92C478F,17)
DAT(11,3,1,0xA07,0x2,0xA202A24F,13)
DAT(11,3,1,0xA13,0x2,0xA202B21B,13)
DAT(11,3,1,0xA7F,0x2,0xA2065417,13)
DAT(11,3,1,0xAA7,0x2,0xA228A86F,15)
DAT(11,3,1,0xAB3,0x2,0xA228B83B,15)
DAT(11,3,1,0xADF,0x2,0xA22C5E37,17)
DAT(11,3,1,0xAEF,0x2,0xA22CEE87,17)
DAT(11,3,1,0xB33,0x2,0xA30213BB,15)
DAT(11,3,1,0xB5F,0x2,0xA306F5B7,19)
DAT(11,3,1,0xB6F,0x2,0xA3064507,13)
DAT(11,3,1,0xB93,0x2,0xA328199B,15)
DAT(11,3,1,0xBAF,0x2,0xA328ABC7,17)
DAT(11,3,1,0xBE7,0x2,0xA32CED2F,19)
DAT(11,3,1,0xC1F,0x2,0xEC02FEF7,21)
DAT(11,3,1,0xC57,0x2,0xEC06B81F,17)
DAT(11,3,1,0xC6B,0x2,0xEC060A43,13)
DAT(11,3,1,0xC73,0x2,0xEC0618FB,17)
DAT(11,3,1,0xC9B,0x2,0xEC285433,15)
DAT(11,3,1,0xCBF,0x2,0xEC28F4D7,19)
DAT(11,3,1,0xCD3,0x2,0xEC2C12DB,17)
DAT(11,3,1,0xCE3,0x2,0xEC2CA26B,17)
DAT(11,3,1,0xD03,0x2,0xED02ED0B,17)
DAT(11,3,1,0xD63,0x2,0xED0609EB,17)
DAT(11,3,1,0xDBB,0x2,0xED28F593,19)
DAT(11,3,1,0xDD7,0x2,0xED2C139F,19)
DAT(11,3,1,0xDE7,0x2,0xED2CA32F,19)
DAT(11,3,1,0xE27,0x2,0xE60246EF,17)
DAT(11,3,1,0xE2B,0x2,0xE6024403,11)
DAT(11,3,1,0xE47,0x2,0xE606A20F,15)
DAT(11,3,1,0xE5F,0x2,0xE606B0B7,17)
DAT(11,3,1,0xEA3,0x2,0xE628EC2B,17)
DAT(11,3,1,0xEBB,0x2,0xE628FE93,19)
DAT(11,3,1,0xECF,0x2,0xE62C0A27,15)
DAT(11,3,1,0xF0B,0x2,0xE702E5A3,17)
DAT(11,3,1,0xF37,0x2,0xE70257FF,21)
DAT(11,3,1,0xF6B,0x2,0xE7060143,13)
DAT(11,3,1,0xF83,0x2,0xE7284D8B,17)
DAT(11,3,1,0xF9B,0x2,0xE7285F33,19)
DAT(11,3,1,0xFD3,0x2,0xE72C19DB,19)
#else
DAT(11,3,2,0x817,0x3,0xC3C0115D,15)
DAT(11,3,2,0x82B,0x3,0xC3C0CFCD,19)
DAT(11,3,2,0x863,0x3,0xC3C59C0D,17)
DAT(11,3,2,0x89F,0x3,0xC3FFD29D,23)
DAT(11,3,2,0x8E7,0x3,0xC3FA5C5D,21)
DAT(11,3,2,0x913,0x3,0xC2C1110D,13)
DAT(11,3,2,0x95B,0x3,0xC2C442CD,15)
DAT(11,3,2,0x97F,0x3,0xC2C48E9D,17)
DAT(11,3,2,0x983,0x3,0xC2FEC00D,17)
DAT(11,3,2,0x9C7,0x3,0xC2FB905D,19)
DAT(11,3,2,0xA07,0x3,0xCFCC005D,17)
DAT(11,3,2,0xA13,0x3,0xCFCC110D,17)
DAT(11,3,2,0xA7F,0x3,0xCFC98E9D,21)
DAT(11,3,2,0xAA7,0x3,0xCFF30C5D,21)
DAT(11,3,2,0xAB3,0x3,0xCFF31D0D,21)
DAT(11,3,2,0xADF,0x3,0xCFF6829D,21)
DAT(11,3,2,0xAEF,0x3,0xCFF65F9D,25)
DAT(11,3,2,0xB33,0x3,0xCECDDD0D,21)
DAT(11,3,2,0xB5F,0x3,0xCEC8429D,17)
DAT(11,3,2,0xB6F,0x3,0xCEC89F9D,21)
DAT(11,3,2,0xB93,0x3,0xCEF2D10D,19)
DAT(11,3,2,0xBAF,0x3,0xCEF20F9D,21)
DAT(11,3,2,0xBE7,0x3,0xCEF75C5D,23)
DAT(11,3,2,0xC1F,0x3,0x9390129D,15)
DAT(11,3,2,0xC57,0x3,0x9395415D,17)
DAT(11,3,2,0xC6B,0x3,0x93959FCD,21)
DAT(11,3,2,0xC73,0x3,0x93958D0D,17)
DAT(11,3,2,0xC9B,0x3,0x93AFD2CD,21)
DAT(11,3,2,0xCBF,0x3,0x93AF1E9D,21)
DAT(11,3,2,0xCD3,0x3,0x93AA810D,15)
DAT(11,3,2,0xCE3,0x3,0x93AA5C0D,17)
DAT(11,3,2,0xD03,0x3,0x9291000D,11)
DAT(11,3,2,0xD63,0x3,0x92949C0D,15)
DAT(11,3,2,0xDBB,0x3,0x92AE1ECD,19)
DAT(11,3,2,0xDD7,0x3,0x92AB815D,17)
DAT(11,3,2,0xDE7,0x3,0x92AB5C5D,19)
DAT(11,3,2,0xE27,0x3,0x9F9CCC5D,21)
DAT(11,3,2,0xE2B,0x3,0x9F9CCFCD,23)
DAT(11,3,2,0xE47,0x3,0x9F99505D,19)
DAT(11,3,2,0xE5F,0x3,0x9F99429D,19)
DAT(11,3,2,0xEA3,0x3,0x9FA30C0D,17)
DAT(11,3,2,0xEBB,0x3,0x9FA31ECD,21)
DAT(11,3,2,0xECF,0x3,0x9FA6939D,21)
DAT(11,3,2,0xF0B,0x3,0x9E9D03CD,19)
DAT(11,3,2,0xF37,0x3,0x9E9DDD5D,23)
DAT(11,3,2,0xF6B,0x3,0x9E989FCD,21)
DAT(11,3,2,0xF83,0x3,0x9EA2C00D,15)
DAT(11,3,2,0xF9B,0x3,0x9EA2D2CD,19)
DAT(11,3,2,0xFD3,0x3,0x9EA7810D,17)
#endif
};

static const struct lfsr_polynom_str poly_table_13[MTGFSR_N13]={
#if (1==MTGFSR_M)
DAT(13,3,1,0x2053,0xA0,0xA0A4F45B,17)
DAT(13,3,1,0x206F,0xA0,0xA0A44607,13)
DAT(13,3,1,0x208B,0xA0,0xA08A0823,11)
DAT(13,3,1,0x20AF,0xA0,0xA08AA8C7,15)
DAT(13,3,1,0x20F3,0xA0,0xA08EFE7B,21)
DAT(13,3,1,0x212F,0xA0,0xA1A00347,13)
DAT(13,3,1,0x2143,0xA0,0xA1A4E54B,17)
DAT(13,3,1,0x216B,0xA0,0xA1A44743,15)
DAT(13,3,1,0x2197,0xA0,0xA18A1BDF,19)
DAT(13,3,1,0x21C7,0xA0,0xA18E4F8F,19)
DAT(13,3,1,0x21DF,0xA0,0xA18E5D37,19)
DAT(13,3,1,0x21E3,0xA0,0xA18EEF6B,21)
DAT(13,3,1,0x21FB,0xA0,0xA18EFDD3,21)
DAT(13,3,1,0x2237,0xA0,0xAAA01AFF,19)
DAT(13,3,1,0x2243,0xA0,0xAAA4EE4B,19)
DAT(13,3,1,0x2297,0xA0,0xAA8A10DF,17)
DAT(13,3,1,0x229B,0xA0,0xAA8A1233,15)
DAT(13,3,1,0x22EF,0xA0,0xAA8EE687,19)
DAT(13,3,1,0x22F7,0xA0,0xAA8EF43F,21)
DAT(13,3,1,0x232B,0xA0,0xABA00903,13)
DAT(13,3,1,0x2347,0xA0,0xABA4EF0F,21)
DAT(13,3,1,0x236F,0xA0,0xABA44D07,17)
DAT(13,3,1,0x2387,0xA0,0xAB8A01CF,17)
DAT(13,3,1,0x23A3,0xA0,0xAB8AA12B,17)
DAT(13,3,1,0x23B7,0xA0,0xAB8AB17F,21)
DAT(13,3,1,0x23BB,0xA0,0xAB8AB393,19)
DAT(13,3,1,0x2413,0xA0,0xE4A0F41B,17)
DAT(13,3,1,0x241F,0xA0,0xE4A0F6F7,21)
DAT(13,3,1,0x2457,0xA0,0xE4A4B01F,17)
DAT(13,3,1,0x24BF,0xA0,0xE48AFCD7,21)
DAT(13,3,1,0x24CB,0xA0,0xE48E0863,15)
DAT(13,3,1,0x24E3,0xA0,0xE48EAA6B,19)
DAT(13,3,1,0x2517,0xA0,0xE5A0F55F,21)
DAT(13,3,1,0x2563,0xA0,0xE5A401EB,17)
DAT(13,3,1,0x2577,0xA0,0xE5A411BF,19)
DAT(13,3,1,0x2587,0xA0,0xE58A4FCF,21)
DAT(13,3,1,0x25AF,0xA0,0xE58AEDC7,21)
DAT(13,3,1,0x25D7,0xA0,0xE58E1B9F,21)
DAT(13,3,1,0x25EB,0xA0,0xE58EA9C3,19)
DAT(13,3,1,0x2603,0xA0,0xEEA0EE0B,19)
DAT(13,3,1,0x267B,0xA0,0xEEA41853,17)
DAT(13,3,1,0x2693,0xA0,0xEE8A549B,19)
DAT(13,3,1,0x26C3,0xA0,0xEE8E00CB,17)
DAT(13,3,1,0x26EB,0xA0,0xEE8EA2C3,19)
DAT(13,3,1,0x2783,0xA0,0xEF8A458B,19)
DAT(13,3,1,0x27C7,0xA0,0xEF8E018F,19)
DAT(13,3,1,0x27CB,0xA0,0xEF8E0363,19)
DAT(13,3,1,0x27DF,0xA0,0xEF8E1337,21)
DAT(13,3,1,0x2807,0xA2,0x08A2084F,13)
DAT(13,3,1,0x281F,0xA2,0x08A21AF7,17)
DAT(13,3,1,0x2823,0xA2,0x08A2A8AB,15)
DAT(13,3,1,0x283B,0xA2,0x08A2BA13,15)
DAT(13,3,1,0x2867,0xA2,0x08A6ECAF,19)
DAT(13,3,1,0x28AB,0xA2,0x08880083,9)
DAT(13,3,1,0x28DF,0xA2,0x088CF437,17)
DAT(13,3,1,0x28FB,0xA2,0x088C54D3,15)
DAT(13,3,1,0x290F,0xA2,0x09A20BE7,17)
DAT(13,3,1,0x292B,0xA2,0x09A2AB03,15)
DAT(13,3,1,0x2993,0xA2,0x0988B39B,17)
DAT(13,3,1,0x29B7,0xA2,0x0988137F,17)
DAT(13,3,1,0x29F3,0xA2,0x098C577B,19)
DAT(13,3,1,0x2A03,0xA2,0x02A2020B,11)
DAT(13,3,1,0x2A0F,0xA2,0x02A200E7,13)
DAT(13,3,1,0x2A6F,0xA2,0x02A6E407,15)
DAT(13,3,1,0x2A7B,0xA2,0x02A6F453,17)
DAT(13,3,1,0x2ADB,0xA2,0x028CFE73,19)
DAT(13,3,1,0x2AEB,0xA2,0x028C4EC3,15)
DAT(13,3,1,0x2AF3,0xA2,0x028C5C7B,17)
DAT(13,3,1,0x2B43,0xA2,0x03A6474B,17)
DAT(13,3,1,0x2B7F,0xA2,0x03A6F517,19)
DAT(13,3,1,0x2B97,0xA2,0x0388B9DF,19)
DAT(13,3,1,0x2C0F,0xA2,0x4CA24EE7,19)
DAT(13,3,1,0x2C2B,0xA2,0x4CA2EE03,17)
DAT(13,3,1,0x2C3F,0xA2,0x4CA2FE57,21)
DAT(13,3,1,0x2CD7,0xA2,0x4C8CB29F,19)
DAT(13,3,1,0x2D2F,0xA2,0x4DA2EF47,21)
DAT(13,3,1,0x2D3B,0xA2,0x4DA2FF13,21)
DAT(13,3,1,0x2D8F,0xA2,0x4D88E567,19)
DAT(13,3,1,0x2DA7,0xA2,0x4D88476F,19)
DAT(13,3,1,0x2DAB,0xA2,0x4D884583,15)
DAT(13,3,1,0x2DE3,0xA2,0x4D8C036B,17)
DAT(13,3,1,0x2E07,0xA2,0x46A2464F,17)
DAT(13,3,1,0x2E5B,0xA2,0x46A610F3,17)
DAT(13,3,1,0x2E8F,0xA2,0x4688EE67,19)
DAT(13,3,1,0x2E97,0xA2,0x4688FCDF,21)
DAT(13,3,1,0x2EAB,0xA2,0x46884E83,15)
DAT(13,3,1,0x2EB3,0xA2,0x46885C3B,17)
DAT(13,3,1,0x2EFB,0xA2,0x468C1AD3,17)
DAT(13,3,1,0x2F17,0xA2,0x47A2575F,21)
DAT(13,3,1,0x2FAF,0xA2,0x47884FC7,19)
DAT(13,3,1,0x2FCF,0xA2,0x478CAB27,19)
DAT(13,3,1,0x2FE7,0xA2,0x478C092F,17)
DAT(13,3,1,0x2FFF,0xA2,0x478C1B97,19)
DAT(13,3,1,0x3007,0xB0,0xB0B0B04F,17)
DAT(13,3,1,0x304F,0xB0,0xB0B4F6A7,21)
DAT(13,3,1,0x3067,0xB0,0xB0B454AF,19)
DAT(13,3,1,0x30D3,0xB0,0xB09E4EDB,21)
DAT(13,3,1,0x30EF,0xB0,0xB09EFC87,21)
DAT(13,3,1,0x310F,0xB0,0xB1B0B3E7,21)
DAT(13,3,1,0x3163,0xB0,0xB1B455EB,21)
DAT(13,3,1,0x31A3,0xB0,0xB19ABB2B,21)
DAT(13,3,1,0x31B7,0xB0,0xB19AAB7F,23)
DAT(13,3,1,0x31EB,0xB0,0xB19EFDC3,23)
DAT(13,3,1,0x31F3,0xB0,0xB19EEF7B,25)
DAT(13,3,1,0x320F,0xB0,0xBAB0B8E7,21)
DAT(13,3,1,0x3227,0xB0,0xBAB01AEF,21)
DAT(13,3,1,0x324B,0xB0,0xBAB4FCE3,23)
DAT(13,3,1,0x3253,0xB0,0xBAB4EE5B,23)
DAT(13,3,1,0x3293,0xB0,0xBA9A009B,17)
DAT(13,3,1,0x329F,0xB0,0xBA9A0277,19)
DAT(13,3,1,0x32B7,0xB0,0xBA9AA07F,21)
DAT(13,3,1,0x32BB,0xB0,0xBA9AA293,19)
DAT(13,3,1,0x32E7,0xB0,0xBA9EF42F,23)
DAT(13,3,1,0x332F,0xB0,0xBBB01947,19)
DAT(13,3,1,0x33AB,0xB0,0xBB9AB383,21)
DAT(13,3,1,0x33C7,0xB0,0xBB9E558F,23)
DAT(13,3,1,0x33E3,0xB0,0xBB9EF56B,25)
DAT(13,3,1,0x341B,0xB0,0xF4B0E6B3,21)
DAT(13,3,1,0x345F,0xB0,0xF4B4A2B7,21)
DAT(13,3,1,0x347B,0xB0,0xF4B40253,17)
DAT(13,3,1,0x3493,0xB0,0xF49A4E9B,21)
DAT(13,3,1,0x34DB,0xB0,0xF49E0873,19)
DAT(13,3,1,0x351F,0xB0,0xF5B0E7F7,25)
DAT(13,3,1,0x354F,0xB0,0xF5B4B3A7,23)
DAT(13,3,1,0x357F,0xB0,0xF5B40317,19)
DAT(13,3,1,0x363B,0xB0,0xFEB04C13,19)
DAT(13,3,1,0x365B,0xB0,0xFEB4A8F3,23)
DAT(13,3,1,0x36CB,0xB0,0xFE9E1263,21)
DAT(13,3,1,0x36D3,0xB0,0xFE9E00DB,21)
DAT(13,3,1,0x36E3,0xB0,0xFE9EB06B,23)
DAT(13,3,1,0x371B,0xB0,0xFFB0EDB3,25)
DAT(13,3,1,0x3747,0xB0,0xFFB4BB0F,25)
DAT(13,3,1,0x3777,0xB0,0xFFB40BBF,25)
DAT(13,3,1,0x37A3,0xB0,0xFF9AF52B,25)
DAT(13,3,1,0x37CF,0xB0,0xFF9E1327,23)
DAT(13,3,1,0x37D7,0xB0,0xFF9E019F,23)
DAT(13,3,1,0x3803,0xB2,0x18B2180B,15)
DAT(13,3,1,0x3817,0xB2,0x18B2085F,17)
DAT(13,3,1,0x3827,0xB2,0x18B2B8EF,21)
DAT(13,3,1,0x384B,0xB2,0x18B65EE3,21)
DAT(13,3,1,0x38A3,0xB2,0x1898122B,15)
DAT(13,3,1,0x38BB,0xB2,0x18980093,13)
DAT(13,3,1,0x38CF,0xB2,0x189CF427,19)
DAT(13,3,1,0x38F3,0xB2,0x189C467B,19)
DAT(13,3,1,0x3907,0xB2,0x19B2194F,19)
DAT(13,3,1,0x390B,0xB2,0x19B21BA3,19)
DAT(13,3,1,0x3967,0xB2,0x19B6FDAF,25)
DAT(13,3,1,0x3983,0xB2,0x1998B38B,19)
DAT(13,3,1,0x39A7,0xB2,0x1998136F,19)
DAT(13,3,1,0x39D3,0xB2,0x199CE7DB,23)
DAT(13,3,1,0x39EF,0xB2,0x199C5587,19)
DAT(13,3,1,0x39F7,0xB2,0x199C473F,21)
DAT(13,3,1,0x3A67,0xB2,0x12B6F6AF,23)
DAT(13,3,1,0x3ABF,0xB2,0x12980AD7,17)
DAT(13,3,1,0x3AD3,0xB2,0x129CECDB,21)
DAT(13,3,1,0x3ADF,0xB2,0x129CEE37,21)
DAT(13,3,1,0x3B3F,0xB2,0x13B2A157,19)
DAT(13,3,1,0x3B53,0xB2,0x13B6475B,21)
DAT(13,3,1,0x3BB7,0xB2,0x1398097F,19)
DAT(13,3,1,0x3BBB,0xB2,0x13980B93,17)
DAT(13,3,1,0x3BE7,0xB2,0x139C5D2F,21)
DAT(13,3,1,0x3BFF,0xB2,0x139C4F97,21)
DAT(13,3,1,0x3C13,0xB2,0x5CB24C1B,19)
DAT(13,3,1,0x3C23,0xB2,0x5CB2FCAB,23)
DAT(13,3,1,0x3C3B,0xB2,0x5CB2EE13,21)
DAT(13,3,1,0x3CA7,0xB2,0x5C98566F,21)
DAT(13,3,1,0x3CC7,0xB2,0x5C9CB28F,21)
DAT(13,3,1,0x3CFB,0xB2,0x5C9C00D3,17)
DAT(13,3,1,0x3D17,0xB2,0x5DB24D5F,23)
DAT(13,3,1,0x3D1B,0xB2,0x5DB24FB3,23)
DAT(13,3,1,0x3E2B,0xB2,0x56B2F403,19)
DAT(13,3,1,0x3E3F,0xB2,0x56B2E457,21)
DAT(13,3,1,0x3ED7,0xB2,0x569CA89F,21)
DAT(13,3,1,0x3F43,0xB2,0x57B6134B,21)
DAT(13,3,1,0x3FAB,0xB2,0x57985F83,21)
DAT(13,3,1,0x3FE3,0xB2,0x579C196B,21)
#else
DAT(13,3,2,0x2053,0xCC,0xCC05410D,15)
DAT(13,3,2,0x206F,0xCC,0xCC059F9D,21)
DAT(13,3,2,0x208B,0xCC,0xCC3FC3CD,23)
DAT(13,3,2,0x20AF,0xCC,0xCC3F0F9D,23)
DAT(13,3,2,0x20F3,0xCC,0xCC3A4D0D,19)
DAT(13,3,2,0x212F,0xCC,0xCD01CF9D,21)
DAT(13,3,2,0x2143,0xCC,0xCD04500D,15)
DAT(13,3,2,0x216B,0xCC,0xCD049FCD,21)
DAT(13,3,2,0x2197,0xCC,0xCD3ED15D,23)
DAT(13,3,2,0x21C7,0xCC,0xCD3B905D,21)
DAT(13,3,2,0x21DF,0xCC,0xCD3B829D,21)
DAT(13,3,2,0x21E3,0xCC,0xCD3B5C0D,21)
DAT(13,3,2,0x21FB,0xCC,0xCD3B4ECD,23)
DAT(13,3,2,0x2237,0xCC,0xC00CDD5D,19)
DAT(13,3,2,0x2243,0xCC,0xC009500D,13)
DAT(13,3,2,0x2297,0xCC,0xC033D15D,19)
DAT(13,3,2,0x229B,0xCC,0xC033D2CD,19)
DAT(13,3,2,0x22EF,0xCC,0xC0365F9D,21)
DAT(13,3,2,0x22F7,0xCC,0xC0364D5D,19)
DAT(13,3,2,0x232B,0xCC,0xC10DCFCD,21)
DAT(13,3,2,0x2347,0xCC,0xC108505D,15)
DAT(13,3,2,0x236F,0xCC,0xC1089F9D,19)
DAT(13,3,2,0x2387,0xCC,0xC132C05D,17)
DAT(13,3,2,0x23A3,0xCC,0xC1320C0D,15)
DAT(13,3,2,0x23B7,0xCC,0xC1321D5D,19)
DAT(13,3,2,0x23BB,0xCC,0xC1321ECD,19)
DAT(13,3,2,0x2413,0xCC,0x9C50110D,15)
DAT(13,3,2,0x241F,0xCC,0x9C50129D,17)
DAT(13,3,2,0x2457,0xCC,0x9C55415D,19)
DAT(13,3,2,0x24BF,0xCC,0x9C6F1E9D,23)
DAT(13,3,2,0x24CB,0xCC,0x9C6A93CD,21)
DAT(13,3,2,0x24E3,0xCC,0x9C6A5C0D,19)
DAT(13,3,2,0x2517,0xCC,0x9D51115D,19)
DAT(13,3,2,0x2563,0xCC,0x9D549C0D,19)
DAT(13,3,2,0x2577,0xCC,0x9D548D5D,21)
DAT(13,3,2,0x2587,0xCC,0x9D6EC05D,21)
DAT(13,3,2,0x25AF,0xCC,0x9D6E0F9D,23)
DAT(13,3,2,0x25D7,0xCC,0x9D6B815D,21)
DAT(13,3,2,0x25EB,0xCC,0x9D6B5FCD,25)
DAT(13,3,2,0x2603,0xCC,0x905C000D,13)
DAT(13,3,2,0x267B,0xCC,0x90598ECD,19)
DAT(13,3,2,0x2693,0xCC,0x9063D10D,17)
DAT(13,3,2,0x26C3,0xCC,0x9066900D,15)
DAT(13,3,2,0x26EB,0xCC,0x90665FCD,21)
DAT(13,3,2,0x2783,0xCC,0x9162C00D,15)
DAT(13,3,2,0x27C7,0xCC,0x9167905D,19)
DAT(13,3,2,0x27CB,0xCC,0x916793CD,21)
DAT(13,3,2,0x27DF,0xCC,0x9167829D,19)
DAT(13,3,2,0x2807,0xCF,0x0FC0005D,17)
DAT(13,3,2,0x281F,0xCF,0x0FC0129D,19)
DAT(13,3,2,0x2823,0xCF,0x0FC0CC0D,19)
DAT(13,3,2,0x283B,0xCF,0x0FC0DECD,23)
DAT(13,3,2,0x2867,0xCF,0x0FC59C5D,23)
DAT(13,3,2,0x28AB,0xCF,0x0FFF0FCD,27)
DAT(13,3,2,0x28DF,0xCF,0x0FFA829D,23)
DAT(13,3,2,0x28FB,0xCF,0x0FFA4ECD,25)
DAT(13,3,2,0x290F,0xCF,0x0EC1039D,19)
DAT(13,3,2,0x292B,0xCF,0x0EC1CFCD,23)
DAT(13,3,2,0x2993,0xCF,0x0EFED10D,23)
DAT(13,3,2,0x29B7,0xCF,0x0EFE1D5D,25)
DAT(13,3,2,0x29F3,0xCF,0x0EFB4D0D,23)
DAT(13,3,2,0x2A03,0xCF,0x03CC000D,15)
DAT(13,3,2,0x2A0F,0xCF,0x03CC039D,19)
DAT(13,3,2,0x2A6F,0xCF,0x03C99F9D,23)
DAT(13,3,2,0x2A7B,0xCF,0x03C98ECD,21)
DAT(13,3,2,0x2ADB,0xCF,0x03F682CD,21)
DAT(13,3,2,0x2AEB,0xCF,0x03F65FCD,25)
DAT(13,3,2,0x2AF3,0xCF,0x03F64D0D,21)
DAT(13,3,2,0x2B43,0xCF,0x02C8500D,15)
DAT(13,3,2,0x2B7F,0xCF,0x02C88E9D,19)
DAT(13,3,2,0x2B97,0xCF,0x02F2D15D,21)
DAT(13,3,2,0x2C0F,0xCF,0x5F90039D,21)
DAT(13,3,2,0x2C2B,0xCF,0x5F90CFCD,25)
DAT(13,3,2,0x2C3F,0xCF,0x5F90DE9D,25)
DAT(13,3,2,0x2CD7,0xCF,0x5FAA815D,23)
DAT(13,3,2,0x2D2F,0xCF,0x5E91CF9D,25)
DAT(13,3,2,0x2D3B,0xCF,0x5E91DECD,25)
DAT(13,3,2,0x2D8F,0xCF,0x5EAEC39D,25)
DAT(13,3,2,0x2DA7,0xCF,0x5EAE0C5D,23)
DAT(13,3,2,0x2DAB,0xCF,0x5EAE0FCD,25)
DAT(13,3,2,0x2DE3,0xCF,0x5EAB5C0D,23)
DAT(13,3,2,0x2E07,0xCF,0x539C005D,19)
DAT(13,3,2,0x2E5B,0xCF,0x539942CD,21)
DAT(13,3,2,0x2E8F,0xCF,0x53A3C39D,23)
DAT(13,3,2,0x2E97,0xCF,0x53A3D15D,23)
DAT(13,3,2,0x2EAB,0xCF,0x53A30FCD,23)
DAT(13,3,2,0x2EB3,0xCF,0x53A31D0D,21)
DAT(13,3,2,0x2EFB,0xCF,0x53A64ECD,23)
DAT(13,3,2,0x2F17,0xCF,0x529D115D,21)
DAT(13,3,2,0x2FAF,0xCF,0x52A20F9D,21)
DAT(13,3,2,0x2FCF,0xCF,0x52A7939D,23)
DAT(13,3,2,0x2FE7,0xCF,0x52A75C5D,23)
DAT(13,3,2,0x2FFF,0xCF,0x52A74E9D,23)
DAT(13,3,2,0x3007,0xDD,0xDD00005D,17)
DAT(13,3,2,0x304F,0xDD,0xDD05539D,23)
DAT(13,3,2,0x3067,0xDD,0xDD059C5D,23)
DAT(13,3,2,0x30D3,0xDD,0xDD3A810D,21)
DAT(13,3,2,0x30EF,0xDD,0xDD3A5F9D,27)
DAT(13,3,2,0x310F,0xDD,0xDC01039D,19)
DAT(13,3,2,0x3163,0xDD,0xDC049C0D,19)
DAT(13,3,2,0x31A3,0xDD,0xDC3E0C0D,21)
DAT(13,3,2,0x31B7,0xDD,0xDC3E1D5D,25)
DAT(13,3,2,0x31EB,0xDD,0xDC3B5FCD,27)
DAT(13,3,2,0x31F3,0xDD,0xDC3B4D0D,23)
DAT(13,3,2,0x320F,0xDD,0xD10C039D,19)
DAT(13,3,2,0x3227,0xDD,0xD10CCC5D,21)
DAT(13,3,2,0x324B,0xDD,0xD10953CD,21)
DAT(13,3,2,0x3253,0xDD,0xD109410D,17)
DAT(13,3,2,0x3293,0xDD,0xD133D10D,21)
DAT(13,3,2,0x329F,0xDD,0xD133D29D,23)
DAT(13,3,2,0x32B7,0xDD,0xD1331D5D,23)
DAT(13,3,2,0x32BB,0xDD,0xD1331ECD,23)
DAT(13,3,2,0x32E7,0xDD,0xD1365C5D,23)
DAT(13,3,2,0x332F,0xDD,0xD00DCF9D,23)
DAT(13,3,2,0x33AB,0xDD,0xD0320FCD,21)
DAT(13,3,2,0x33C7,0xDD,0xD037905D,21)
DAT(13,3,2,0x33E3,0xDD,0xD0375C0D,21)
DAT(13,3,2,0x341B,0xDD,0x8D5012CD,19)
DAT(13,3,2,0x345F,0xDD,0x8D55429D,21)
DAT(13,3,2,0x347B,0xDD,0x8D558ECD,23)
DAT(13,3,2,0x3493,0xDD,0x8D6FD10D,23)
DAT(13,3,2,0x34DB,0xDD,0x8D6A82CD,21)
DAT(13,3,2,0x351F,0xDD,0x8C51129D,19)
DAT(13,3,2,0x354F,0xDD,0x8C54539D,21)
DAT(13,3,2,0x357F,0xDD,0x8C548E9D,21)
DAT(13,3,2,0x363B,0xDD,0x815CDECD,23)
DAT(13,3,2,0x365B,0xDD,0x815942CD,19)
DAT(13,3,2,0x36CB,0xDD,0x816693CD,21)
DAT(13,3,2,0x36D3,0xDD,0x8166810D,17)
DAT(13,3,2,0x36E3,0xDD,0x81665C0D,19)
DAT(13,3,2,0x371B,0xDD,0x805D12CD,19)
DAT(13,3,2,0x3747,0xDD,0x8058505D,17)
DAT(13,3,2,0x3777,0xDD,0x80588D5D,19)
DAT(13,3,2,0x37A3,0xDD,0x80620C0D,15)
DAT(13,3,2,0x37CF,0xDD,0x8067939D,21)
DAT(13,3,2,0x37D7,0xDD,0x8067815D,19)
DAT(13,3,2,0x3803,0xDE,0x1EC0000D,15)
DAT(13,3,2,0x3817,0xDE,0x1EC0115D,19)
DAT(13,3,2,0x3827,0xDE,0x1EC0CC5D,21)
DAT(13,3,2,0x384B,0xDE,0x1EC553CD,23)
DAT(13,3,2,0x38A3,0xDE,0x1EFF0C0D,23)
DAT(13,3,2,0x38BB,0xDE,0x1EFF1ECD,27)
DAT(13,3,2,0x38CF,0xDE,0x1EFA939D,25)
DAT(13,3,2,0x38F3,0xDE,0x1EFA4D0D,23)
DAT(13,3,2,0x3907,0xDE,0x1FC1005D,19)
DAT(13,3,2,0x390B,0xDE,0x1FC103CD,21)
DAT(13,3,2,0x3967,0xDE,0x1FC49C5D,23)
DAT(13,3,2,0x3983,0xDE,0x1FFEC00D,23)
DAT(13,3,2,0x39A7,0xDE,0x1FFE0C5D,25)
DAT(13,3,2,0x39D3,0xDE,0x1FFB810D,23)
DAT(13,3,2,0x39EF,0xDE,0x1FFB5F9D,29)
DAT(13,3,2,0x39F7,0xDE,0x1FFB4D5D,27)
DAT(13,3,2,0x3A67,0xDE,0x12C99C5D,21)
DAT(13,3,2,0x3ABF,0xDE,0x12F31E9D,23)
DAT(13,3,2,0x3AD3,0xDE,0x12F6810D,19)
DAT(13,3,2,0x3ADF,0xDE,0x12F6829D,21)
DAT(13,3,2,0x3B3F,0xDE,0x13CDDE9D,25)
DAT(13,3,2,0x3B53,0xDE,0x13C8410D,17)
DAT(13,3,2,0x3BB7,0xDE,0x13F21D5D,23)
DAT(13,3,2,0x3BBB,0xDE,0x13F21ECD,23)
DAT(13,3,2,0x3BE7,0xDE,0x13F75C5D,25)
DAT(13,3,2,0x3BFF,0xDE,0x13F74E9D,25)
DAT(13,3,2,0x3C13,0xDE,0x4E90110D,17)
DAT(13,3,2,0x3C23,0xDE,0x4E90CC0D,19)
DAT(13,3,2,0x3C3B,0xDE,0x4E90DECD,23)
DAT(13,3,2,0x3CA7,0xDE,0x4EAF0C5D,23)
DAT(13,3,2,0x3CC7,0xDE,0x4EAA905D,21)
DAT(13,3,2,0x3CFB,0xDE,0x4EAA4ECD,23)
DAT(13,3,2,0x3D17,0xDE,0x4F91115D,21)
DAT(13,3,2,0x3D1B,0xDE,0x4F9112CD,21)
DAT(13,3,2,0x3E2B,0xDE,0x429CCFCD,23)
DAT(13,3,2,0x3E3F,0xDE,0x429CDE9D,23)
DAT(13,3,2,0x3ED7,0xDE,0x42A6815D,19)
DAT(13,3,2,0x3F43,0xDE,0x4398500D,17)
DAT(13,3,2,0x3FAB,0xDE,0x43A20FCD,21)
DAT(13,3,2,0x3FE3,0xDE,0x43A75C0D,21)
#endif
};

/**************** Frogbit key schedule ****************/

/*
 * A static table is used as a blueprint for PRNG state initialization.
 * Polynom constants will be ANDed to the hexadecima constants below
 * according to the MTGFSR principles, see ECRYPT_init function.
 */
static ECRYPT_AE_frogbit_prng def_prng[FROGBIT10]=
{{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
};

/*
 * Key and message independent initialization. According to ECRYPT NoE
 * API requirements.
 */
void ECRYPT_init(void)
{
   int ik;
   for (ik=0;ik<FROGBIT10;ik++) {
      int i;
      u32 p0, p1;
      u32 combi;
      combi=0;
      if (!(ik&1)) {
         p0=poly_table_13[prng_design[ik][0]].Aj;
         p1=poly_table_11[prng_design[ik][1]].Aj;
         for (i=0;i<13;i++) {
            combi|=(0!=(p0&(1<<i)))<<(2*13-2-2*i);
         }
         for (i=0;i<11;i++) {
            combi|=(0!=(p1&(1<<i)))<<(2*11-1-2*i);
         }
      }
      else {
         p0=poly_table_11[prng_design[ik][0]].Aj;
         p1=poly_table_13[prng_design[ik][1]].Aj;
         for (i=0;i<11;i++) {
            combi|=(0!=(p0&(1<<i)))<<(2*11-2-2*i);
         }
         for (i=0;i<13;i++) {
            combi|=(0!=(p1&(1<<i)))<<(2*13-1-2*i);
         }
      }
      for (i=1;i<(1<<MTGFSR_F);i++) {
         def_prng[ik].table[i]&=combi;
      }
   }
}

/*
 * These parts of the key schedule:
 * "Nibble Distribution to First State Word in Each PRNG"
 * "Other State Words in Each PRNG"
 */
int ECRYPT_frogbit_key_shed(
  ECRYPT_AE_frogbit_prng *prng,
  int ivsize,
  const u8* key)
{
   int ik;
   int i;
   memcpy(prng,def_prng,sizeof(def_prng));

   /* "Nibble Distribution to First State Word in Each PRNG" */
   switch (ivsize) {
   case 192:
      for (ik=0;ik<FROGBIT10;ik++) {
         prng[ik].bits[0]=(key[2*ik+1]<<8)||key[2*ik];
      }
      ik=2*FROGBIT10;
      break;
   case 256:
      for (ik=0;ik<FROGBIT10/2;ik++) {
         prng[2*ik  ].bits[0]=  ((key[5*ik+2]&0x0F)<<16)
                              || (key[5*ik+1]<<8)
                              ||  key[5*ik];
         prng[2*ik+1].bits[0]=   (key[5*ik+4]<<12)
                              || (key[5*ik+3]<<4)
                              || (key[5*ik+2]>>4);
      }
      ik=5*FROGBIT10/2;
      break;
   default:
     ik=0;
   }

   /* "Other State Words in Each PRNG" */
   for (i=0;i<FROGBIT10;i++) {
      prng[i].bits[1]=prng[i].bits[0]^FROGBIT_KEYSCHED_C1;
      prng[i].bits[2]=prng[i].bits[0]+FROGBIT_KEYSCHED_C2;
   }
   return ik;
}

#if TEST

/*
 * Berlekamp-Massey validation of the above polynom tables:
 *  1) initialize the PRNG as in the case fo the Frogbit reference
 *     implementation,
 *  2) get output bits from the individual TGFSR generator components
 *  3) pass such output sequence to the Berlekamp-Massey algorithm
 *  4) verify that the equivalent LFSR generating polynomial has been
 *     recovered by the Berlekamp-Massey algorithm
 *  5) get output bit pairs from the combined MTGFSR
 *  6) pass such output sequence to the Berlekamp-Massey algorithm
 *  7) report the recovered polynomial (informative only)
 *  8) get output bits as the exclusive-or of the two bits from each step of
 *     the combined MTGFSR
 *  9) pass such output sequence to the Berlekamp-Massey algorithm
 * 10) report the recovered polynomial (informative only)
 *
 * The reported values in steps 7 and 10 agree with the expected MTGFSR
 * generator period.
 *
 * The steps 1-10 are repeated so that every entries in the above tables are
 * validated.
 *
 * Sample output portion:

 44: A0EF8A458B -- step 4
 12: 2A2065417 -- step 4
1001055150151044501500001001151500001 -- step 7
104F71D2B1C0105DC01 -- step 10

 */

int main(int argc, char *argv[])
{
static unsigned char data[1000];
static llfsr_prng bit;
static ECRYPT_AE_frogbit_prng test_prng[FROGBIT10];

static ECRYPT_AE_frogbit_prng def0_prng[FROGBIT10]=
{{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x01555555,0x002AAAAA,0x017FFFFF}}
,{MTGFSR_N-1,{0,0,0},{0,0x00155555,0x02AAAAAA,0x02BFFFFF}}
};

  int it, ik, ib, i;

  for (it=0;it<MTGFSR_N13;it+=FROGBIT10) {
   /* First test the default design, then try every
      PRNG parameters, 10 at a time */

   /*  1) initialize the PRNG as in the case fo the Frogbit reference
          implementation */
   ECRYPT_init();
   memset(data,0x69,sizeof(data));
   ECRYPT_frogbit_key_shed(test_prng,256,data);

   for (ik=0;ik<FROGBIT10;ik++) {
      int iii;
      for (ib=0;ib<2;ib++) {
         /* Test a single bit from the two-bit generator output */
         for (i=0;i<1000;i++) {
            data[i]=0;
            for (iii=0;iii<CHAR_BIT;iii++) {
               int s;
               /*  2) get output bits from the individual TGFSR generator
                      components */
               MTGFSR_DRAW(test_prng+ik,s)
               data[i]|=(1&(s>>ib))<<iii;
            }
         }
         /*  3) pass such output sequence to the Berlekamp-Massey algorithm */
         bit.berlekamp_massey(data,1000*CHAR_BIT);

         /*  4) verify that the equivalent LFSR generating polynomial has been
                recovered by the Berlekamp-Massey algorithm */
         fprintf(stdout,"%3d: ",prng_design[ik][ib]);
         bit.fprintf(stdout);
         fprintf(stdout,"\n");
         if ((!(ik&1))^ib) {
            if (  (bit.terms[1]!=poly_table_13[prng_design[ik][ib]].Bj[1])
                ||(bit.terms[0]!=poly_table_13[prng_design[ik][ib]].Bj[0])
               ) {
               fprintf(stdout,"Error 0x%X%8.8X 0x%X%8.8X\n"
                  ,bit.terms[1],bit.terms[0]
                  ,poly_table_13[prng_design[ik][1-ib]].Bj[1]
                  ,poly_table_13[prng_design[ik][1-ib]].Bj[0]);
            }
         }
         else {
            if (  (bit.terms[1]!=poly_table_11[prng_design[ik][ib]].Bj[1])
                ||(bit.terms[0]!=poly_table_11[prng_design[ik][ib]].Bj[0])
               ) {
               fprintf(stdout,"Error 0x%X%8.8X 0x%X%8.8X\n"
                  ,bit.terms[1],bit.terms[0]
                  ,poly_table_11[prng_design[ik][ib]].Bj[1]
                  ,poly_table_11[prng_design[ik][ib]].Bj[0]);
            }
         }
      }

      /* Try the Berlekamp-Massey algo on the two-bit generator output per se */
      for (i=0;i<1000;i++) {
         data[i]=0;
         for (iii=0;iii<CHAR_BIT/2;iii++) {
            int s;
            /*  5) get output bit pairs from the combined MTGFSR */
            MTGFSR_DRAW(test_prng+ik,s)
            data[i]|=(1&(s   ))<<(2*iii  );
            data[i]|=(1&(s>>1))<<(2*iii+1);
         }
      }
      /*  6) pass such output sequence to the Berlekamp-Massey algorithm */
      bit.berlekamp_massey(data,1000*CHAR_BIT);
      /*  7) report the recovered polynomial (informative only) */
      bit.fprintf(stdout);fprintf(stdout,"\n");

      /* Try the Berlekamp-Massey algo on the "exclusive or" of the two-bit
            generator output */
      for (i=0;i<1000;i++) {
         data[i]=0;
         for (iii=0;iii<CHAR_BIT;iii++) {
            int s;
            /*  8) get output bits as the exclusive-or of the two bits from
                   each step of the combined MTGFSR */
            MTGFSR_DRAW(test_prng+ik,s)
            data[i]|=(1&(s^(s>>1)))<<iii;
         }
      }
      /*  9) pass such output sequence to the Berlekamp-Massey algorithm */
      bit.berlekamp_massey(data,1000*CHAR_BIT);
      /* 10) report the recovered polynomial (informative only) */
      bit.fprintf(stdout);fprintf(stdout,"\n");
   }

   /* make a new batch of PRNG designs */
   for (ik=0;ik<FROGBIT10;ik++) {
      /* Fix a static initialization that is destroyed in the PRNG
         initialization */
      memcpy(def_prng,def0_prng,sizeof(def0_prng));

      if (!(ik&1)) {
         prng_design[ik][0]=(it+ik)%MTGFSR_N13;
         prng_design[ik][1]=(it+ik)%MTGFSR_N11;
      }
      else {
         prng_design[ik][0]=(it+ik)%MTGFSR_N11;
         prng_design[ik][1]=(it+ik)%MTGFSR_N13;
      }
   }
  }
}
#endif

