#include <stdio.h>
#include <gmp.h>
#include "error.h"
#include "snefrufile.h"

void usage()
{
  fprintf(stderr,"sign: usage: sign secretfile\n");
  exit(100);
}
void die_read(fn)
char *fn;
{
  fprintf(stderr,"sign: fatal: unable to read %s: %s\n",fn,error_str(errno));
  exit(111);
}
void die_check()
{
  fprintf(stderr,"sign: fatal: internal consistency check failed\n");
  exit(100);
}

void sign(s,t,h,q,r,n,x,y)
mpz_t s;
mpz_t t;
mpz_t h;
mpz_t q;
mpz_t r;
mpz_t n;
mpz_t x;
mpz_t y;
{
  mpz_t a;
  mpz_t b;

  mpz_init(a);
  mpz_init(b);

  mpz_add_ui(a,q,1);
  mpz_tdiv_q_ui(a,a,4);
  mpz_powm(a,h,a,q);

  mpz_add_ui(b,r,1);
  mpz_tdiv_q_ui(b,b,4);
  mpz_powm(b,h,b,r);

  mpz_mul(a,a,r);
  mpz_mul(a,a,y);
  mpz_mul(b,b,q);
  mpz_mul(b,b,x);
  mpz_add(s,a,b);
  mpz_mod(s,s,n);

  mpz_add(a,s,s);
  if (mpz_cmp(a,n) < 0)
    mpz_sub(s,n,s); /* to guarantee t > 0 */

  mpz_mul(b,s,s);
  mpz_sub(b,b,h);
  mpz_tdiv_qr(t,a,b,n);
  if (mpz_sgn(a) != 0) die_check();

  mpz_clear(b);
  mpz_clear(a);
}

snefrufile sf;
unsigned char data[80] = {
  1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  3,1,4,1,5,9,2,6,5,3,5,8,9,7,9,3,2,3,8,4,6,2,6,4,3,3,8,3,2,7,9,5
} ;

void main(argc,argv)
int argc;
char **argv;
{
  mpz_t q;
  mpz_t r;
  mpz_t n;
  mpz_t x;
  mpz_t y;
  mpz_t h;
  mpz_t s;
  mpz_t t;
  char *fnsecret;
  FILE *fisecret;
  int c;
  int i;
  int nlen;

  snefrufile_clear(&sf);
  snefrufile_addn(&sf,"djb-sigs-",9);
  while ((c = getchar()) != EOF)
    snefrufile_add(&sf,c);
  snefrufile_hash(&sf,data + 16);

  putchar(0); /* extensibility */
  for (i = 16;i < 48;++i)
    putchar(data[i]);

  mpz_init(q);
  mpz_init(r);
  mpz_init(n);
  mpz_init(x);
  mpz_init(y);
  mpz_init(s);
  mpz_init(t);
  mpz_init(h);

  fnsecret = argv[1];
  if (!fnsecret) usage();

  fisecret = fopen(fnsecret,"r");
  if (!fisecret) die_read(fnsecret);
  if (mpz_inp_raw(q,fisecret) == 0) die_read(fnsecret);
  if (mpz_inp_raw(r,fisecret) == 0) die_read(fnsecret);
  if (mpz_inp_raw(x,fisecret) == 0) die_read(fnsecret);
  if (mpz_inp_raw(y,fisecret) == 0) die_read(fnsecret);
  fclose(fisecret);

  mpz_mul(n,q,r);
  nlen = (mpz_sizeinbase(n,2) + 63) / 64;

  mpz_set_ui(h,0);
  for (i = sizeof data - 1;i >= 0;--i) {
    mpz_mul_2exp(h,h,8);
    mpz_add_ui(h,h,(unsigned long) data[i]);
  }

  mpz_mul_2exp(h,h,64 * nlen);

  if (mpz_jacobi(h,r) == -1) {
    mpz_neg(h,h);
    putchar('-');
  }
  else
    putchar('+');
  if (mpz_jacobi(h,q) == -1) {
    mpz_add(h,h,h);
    putchar('2');
  }
  else
    putchar('1');

  sign(s,t,h,q,r,n,x,y);

  for (i = 0;i < nlen * 8;++i) {
    putchar(mpz_get_ui(s));
    mpz_tdiv_q_2exp(s,s,8);
  }
  for (i = 0;i < nlen * 8;++i) {
    putchar(mpz_get_ui(t));
    mpz_tdiv_q_2exp(t,t,8);
  }

  exit(0);
}
